<?php
/* 
 * PvtContent Mail Actions bridge to Mailchimp
 * uses WP HTTP APIs and Mailchimp API v3 
 * 
 * @author :	Luca Montanari (LCweb)
 * @website:	http://www.lcweb.it
 */
 
class pcma_mailchimp {
	
    private $debug_mode;
    
	private $api_key; // api key given by customer
	private $list; // target list to sync users in
	private $mc_baseurl; // baseurl for calls - depends on APIKEY datacenter
	
	private $new_on_pending; // whether to sync new users on pending status
	
	public $call_error = ''; // last error found in a call 
    public $batch_status; 
	public $synced_num = 0; // number of synced users
	
	
	// setup basic properties
	public function __construct() {     
        $this->debug_mode = (isset($_GET['pcma_mc_debug'])) ? true : false;
        
		$this->api_key 			= get_option('pcma_mc_apikey', '');
		$this->list 			= get_option('pcma_mc_list', ''); 
		$this->new_on_pending	= get_option('pcma_mc_pending'); 
	}


	/* perform an HTTP POST call 
	 *
	 * @param (string) $url - url to call
	 * @param (string) $method - POST or GET or PATCH or PUT or DELETE
	 * @param (array) $args - associative parameters array to attach to the call
	 * @param (array) $headers - associative array for custom call headers
	 *
	 * @return (mixed) call answer or false if call failed
	 */
	private function call($url, $method = 'POST', $args = array(), $headers = array()) {

		// prepare authorization data in header
		$headers['Authorization'] = "Basic ". base64_encode('anystring:'. $this->api_key);
		
		// if POST - is a json submission
		if($method == 'POST' || $method == 'PATCH' || $method == 'PUT') {
			$headers['Content-Type'] = 'application/json';	
			$args = json_encode($args);
		}
        
		// compose
		$data = @wp_remote_post($url, array(
			'headers'	=> $headers,
			'body' 		=> $args,
			'method'	=> $method
		));
        
        // DELETE command returns status 204 as success 
        if($method == 'DELETE' && wp_remote_retrieve_response_code($data) == 204) {
            return true;    
        }

        // errors hendling
		if(is_wp_error($data) || 200 != wp_remote_retrieve_response_code($data) || empty($data['body'])) {

            $this->call_error = 'HTTP Status '. wp_remote_retrieve_response_code($data) .' '; 
            
			// save error for debug
			if(is_wp_error($data)) {
				$this->call_error .= $data->get_error_message();
			}
			else if(!empty($data['body'])) {
				$arr = json_decode($data['body'], true);
				
				if(isset($arr['detail'])) {
					$this->call_error = $arr['detail'] .' ';
					
					if(isset($arr['errors'])) {
                        foreach($arr['errors'] as $err) {
                            $this->call_error .= json_encode($err);        
                        }
					}
				}
			}
            
			return false; 
		}
        
        $response = json_decode(wp_remote_retrieve_body($data), true);

        
        // even with batch 401 error... command is executed when retrieving batch status.. bah
        if(strpos($url, '/batches') !== false && !$this->get_batch_status($response)) {
            //return false;    
        }
		return $response;	 
	}


    
    /* get batch process status */
    private function get_batch_status($response) {
        $this->batch_status = false;
        
        if(!isset($response['_links'])) {
            return true;    
        }
        
        $status_url = $response['_links'][1]['href']; 
        $response = wp_remote_get($status_url);
 
        if(!is_array($response) ||is_wp_error($response)) {
            return false;
        }
        
        $body = json_decode($response['body'], true);
        
        if((int)$body['status'] != 200) {
            $this->call_error = 'Batch error '. $body['status'] .' - '. $body['detail'] .' <a href="'. esc_attr($status_url) .'" target="_blank">(LINK)</a>';  
            return false;
        }
        
        return true;
    }
    
    
    
    
	/* checks apikey & list ID existence and setup baseurl
	 * is MANDATORY to all it before any other opereation
	 *
	 * @param (bool) $only_apikey - whether to check only apikey existence (eg. to get lists)
	 * @return (bool)
	 */
	public function is_ready($only_apikey = false) {
		if(empty($this->api_key) || (!$only_apikey && empty($this->list))) {
			return false;	
		}

		// setup baseurl
		$arr = explode('-', $this->api_key);
		$this->mc_baseurl = 'https://'. end($arr) .'.api.mailchimp.com/3.0/';

		return true;
	}


	
	// utility - returns call response or error message
	private function debug_call($response) {
		if($response) {
			var_dump($response);	
		} else {
			var_dump( $this->call_error );
		}
	}


	/***********************************************************************************************************************/	
		


	/* Get user fields to retrieve (first/last names and e-mail are excluded) */
	private function user_fields() {
		include_once(PC_DIR .'/classes/pc_form_framework.php');
		
		$f_fw = new pc_form;
		$fields = $f_fw->fields;
		
		$to_remove = array('name', 'surname', 'psw', 'categories', 'email', 'pc_disclaimer', 'pcma_mc_disclaimer');
		foreach($to_remove as $tr) {
			unset( $fields[ $tr ] );	
		}
		
		return $fields;
	}



	/* Get users data and categories to allow mailchimp sync
	 *
	 * @param (array|string) $who_to_get - could be "any" or an array of users ID to get
	 * @return (array) query result
	 */
	private function users_data($who_to_get) {
		global $pc_users;
		
		$to_get = array('id', 'insert_date', 'name', 'surname', 'email', 'categories');
		if(get_option('pcma_mc_sync_all')) {
			$to_get = array_unique( array_merge($to_get, array_keys($this->user_fields()) ));	
		}
		
		$args = array(
			'to_get' 	=> $to_get,
			'limit'		=> -1,
			'search' 	=> array(
				array('key'=>'email', 'operator'=>'!=', 'val'=>''),
				array('key'=>'pcma_mc_disclaimer', 'operator'=>'=', 'val'=>1)
			)
		);
		
		// specify user IDs 
		if($who_to_get != 'any') {
			$args['search'][] = array('key'=>'id', 'operator'=>'IN', 'val'=>(array)$who_to_get);
		}
		
		// otherwise fetch only active users
		else {
			$args['status']	= 1;
		}
		
		$user_query = $pc_users->get_users($args);
		
		if(!is_array($user_query) || !count($user_query)) {
            return array();
        }
		else {
			$to_sync = array();
			$existing_mails = array();

			// avoid doubled e-mails
			foreach($user_query as $user) {
				if(filter_var($user['email'], FILTER_VALIDATE_EMAIL) && !in_array($user['email'], $existing_mails)) {
					$to_sync[ $user['id'] ] = $user; 
					$existing_mails[] = $user['email'];	
				}
			}
			
            
            // PCMA-FILTER - Allows further filter over users needing Mailchimp sync. Passes array(user_id => user_data)
            return (array)apply_filters('pcma_mc_to_sync_array', $to_sync);
		}
	}

	

	/***********************************************************************************************************************/
	
	
	
	
	/* get mailchimp lists! */
	public function get_lists() {
		$url = $this->mc_baseurl. 'lists';
		$data = $this->call($url, 'GET', array(
			'count' => 999
		));

		if(!is_array($data) || !isset($data['lists']) || empty($data['lists'])) {
			return array();	
		} else {
			return $data['lists'];
		}	
	}


	
	/* 
	 * check interests-category (cats wrapper) existence and eventually creates it 
	 * @return false if error is found or the element ID
	 */
	public function check_grouping() {
		$name = 'PrivateContent Categories';
		
		$url = $this->mc_baseurl. 'lists/'. $this->list .'/interest-categories';
		$response = $this->call($url, 'GET', array(
			'count' => 999
		));
		
		if(!$response) {
            return false;
        }
		else {
			foreach((array)$response['categories'] as $cat) {
				if($cat['title'] == $name) {
					update_option('pcma_interests_cat', $cat['id']);
					return $cat['id'];	
				}
			}
						
			// not existing - create it
			$args = array(
				'title' => $name,
				'type'	=> 'checkboxes'
			);
			$url = $this->mc_baseurl. 'lists/'. $this->list .'/interest-categories';
			$response = $this->call($url, 'POST', $args);
			
			// store in DB
			if(is_array($response)) {
				update_option('pcma_interests_cat', $response['id']);
				return $response['id'];	
			}	
		}
	}

	
	
	/* 
	 * sync categories (add/update/remove)
	 * sync adding ID into the name to match everything easier
	 *
	 * @param (bool) $save_in_db - flag to be used calling function recursively - creates 'pcma_interests' option -> associative array(wp_id => mc_id)
	 * @return (bool)
	 */
	public function sync_cats($save_in_db = false) {

		// wrapper exists?
		$wrapper_id = $this->check_grouping();
		if(!$wrapper_id) {
            return false;
        }
		
		// get categories
		$wp_cats = pc_static::user_cats();
		$to_add = $wp_cats; // clone array to know which ones must be added as new
		if(empty($wp_cats)) {
            return true;
        }
		
		// check already synced categories
		$get_interests_url = $this->mc_baseurl. '/lists/'. $this->list .'/interest-categories/'. $wrapper_id .'/interests';
		$response = $this->call($get_interests_url, 'GET', array(
			'count' => 999
		));

		
		// if just have to save into DB
		if($save_in_db) {
			if(!$response) {
                return false;
            }
			$to_store = array();
			
			foreach($wp_cats as $wp_id => $wp_name) {
				foreach($response['interests'] as $int) {
					
					$to_match = $wp_name .' (#'. $wp_id .')';
					
					if($int['name'] == $to_match) {	
						$to_store[ $wp_id ] = $int['id'];
					}
				}
			}
			
			if(empty($to_store)) {
				$this->sync_cats();	
			}
			
			update_option('pcma_interests', $to_store);
			return true;
		}

		
		// normal execution - add/update/remove
		if(!$response) {
            return false;
        }
		else {
			$synced = array();
			
			foreach($response['interests'] as $int) {
				$arr = explode('(#', $int['name']);

				$synced[ $int['id'] ] = array(
					'name' 	=> trim($arr[0]),
					'wp_id' => (int)substr($arr[1], 0, -1)
				);		
			}	
		}
		
		// match with website ones to know if there are outdated/deleted ones
		$to_update = array();
		$to_remove = array();
		
		foreach($synced as $mc_id => $data) {
			
			if(!isset( $wp_cats[ $data['wp_id'] ] )) {
				$to_remove[] = $mc_id;
			}	
			
			else {
				$wp_name = $wp_cats[ $data['wp_id'] ];
				if($wp_name != $data['name']) {
					$to_update[$mc_id] = $data['wp_id'];	
				}

				unset($to_add[ $data['wp_id'] ]); // exists - do not add
			}
		}
		
		
		// setup batch operation
		if(count($to_update) || count($to_remove) || count($to_add)) {
			$structure = array('operations' => array());
		
			// to update
			foreach($to_update as $mc_id => $wp_id) {
				 $structure['operations'][] = array(
					'path' => 'lists/'. $this->list .'/interest-categories/'. $wrapper_id .'/interests/'. $mc_id,
					'method' => 'PATCH',
					'body' => json_encode(array(
						'name' => $wp_cats[$wp_id] .' (#'. $wp_id .')'
					))
				);	
			}

			// to remove
			foreach($to_remove as $mc_id) {
				 $structure['operations'][] = array(
					'path' => 'lists/'. $this->list .'/interest-categories/'. $wrapper_id .'/interests/'. $mc_id,
					'method' => 'DELETE',
				);	
			}
			
			// to add
			foreach($to_add as $wp_id => $wp_name) {
				 $structure['operations'][] = array(
					'path' => 'lists/'. $this->list .'/interest-categories/'. $wrapper_id .'/interests',
					'method' => 'POST',
					'body' => json_encode(array(
						'name' => $wp_cats[$wp_id] .' (#'. $wp_id .')'
					))
				);	
			}
			
			$url = $this->mc_baseurl. 'batches';
			$response = $this->call($url, 'POST', $structure);
			
			// successfully performed - refetch to update DB record (wp_id => mc_id)
			return ($response) ? $this->sync_cats(true) : false;
		}
		
		// secutity check - if DB is empty - populate it
		if(!get_option('pcma_interests')) {
			return ($response) ? $this->sync_cats(true) : false;	
		}
		
		return true;
	}
	


	/* 
	 * Sync fields that are extra on mailchimp (add/update/remove)
	 * @return (bool)
	 */
	public function sync_fields() {
		
		// get fields
		$pc_fields = $this->user_fields();
		$to_add = $pc_fields; // clone array to know which ones must be added as new
		
		
		// check already synced categories
		$url = $this->mc_baseurl. '/lists/'. $this->list .'/merge-fields';
		$response = $this->call($url, 'GET');
		
		if(!$response) {
            return false;
        }
		else {
			$existing = array();
			
			foreach($response['merge_fields'] as $mf) {
				if($mf['tag'] == 'FNAME' || $mf['tag'] == 'LNAME') {continue;} // ignore default MC fields
				
				$existing[ $mf['tag'] ] = array(
					'id'	=> $mf['merge_id'],
					'name' 	=> $mf['name']
				); 
			}	
		}
		

		// match with website ones to know if there are outdated ones
		$to_update = array();
		
		foreach($existing as $tag => $data) {
			if(isset($pc_fields[$tag])) {
				
				if($data['name'] != $pc_fields[$tag]['label']) {
					$to_update[ $tag ] = $data;	
				}
				
				unset($to_add[ $tag ]); // exists - do not add
			}
		}
		
		// setup batch operation
		if(count($to_update) || count($to_add)) {
			$structure = array('operations' => array());
		
			// to update
			foreach($to_update as $tag => $data) {
				 $structure['operations'][] = array(
					'path' => 'lists/'. $this->list .'/merge-fields/'. $data['id'],
					'method' => 'PATCH',
					'body' => json_encode(array(
						'name' 	=> $pc_fields[$tag]['label'],
						'type' 	=> 'text',
						'tag' 	=> $tag
					))
				);	
			}

			// to add
			foreach($to_add as $tag => $data) {
				 $structure['operations'][] = array(
					'path' => 'lists/'. $this->list .'/merge-fields',
					'method' => 'POST',
					'body' => json_encode(array(
						'name' 	=> $data['label'],
						'type' 	=> 'text',
						'tag' 	=> $tag
					))
				);	
			}
			
			$url = $this->mc_baseurl. 'batches';
			$data = $this->call($url, 'POST', $structure);
			
            if($this->debug_mode) {
			     $this->debug_call($data);
            }
			return ($data) ? true : false;
		}
		
		return true;
	}



	/* 
	 * suscribe or update members (including categories and custom fields)
	 *
	 * @param (array|string) $subj - use "any" to sync any user or an array of users ID
	 * @pram (bool) $resync_cats - whether to force categories sync (sort of resetter to be sure everything will work)
	 *
	 * @return (bool)
	 */
	public function subscribe_members($subj = 'any', $resync_cats = false) {
		if(!is_array($subj) && $subj != 'any') {
			$subj = (array)$subj;	
		}
		
		global $pc_users;
		$to_sync = $this->users_data($subj); 
		$this->synced_num = 0;
		
		// no users to sync
		if(!count($to_sync)) {
			if($subj == 'any') {
                return true;
            }
            else {
                $this->call_error = 'user not found';
                return false;
            }
		}
		
		// v1.62 - fix bug forcing synced cats cleaning
		if(!get_option('pcma_v1.62_mc_fix')) {
			delete_option('pcma_interests');
			update_option('pcma_v1.62_mc_fix', 1);	
		}
		
		
		// retrieve cats association with MC
		$cats = get_option('pcma_interests', array());
		if(!is_array($cats) || empty($cats) || $resync_cats) {
			if($this->sync_cats(true)) {
				$cats = get_option('pcma_interests', array());	
			}
            else {
                return false;    
            }
		}
		
		// create batch structure 
		$structure = array('operations' => array());
		foreach($to_sync as $ud) {
			
			// PCMA-FILTER - allow remote control over single user sync with mailchimp - return true to sync or false to unsync - passes user ID
			$to_subscribe_or_not = (bool)apply_filters('pcma_mailchimp_user_subscribe', true, $ud['id']); 
			if(!$to_subscribe_or_not) {
				continue;	
			}
			
			// merge fields
			$merge_fields = array(
				'FNAME' => $ud['name'],
				'LNAME' => $ud['surname']
			);
			foreach($ud as $index => $val) {
				if(in_array($index, array('email', 'name', 'surname', 'insert_date', 'categories'))) {
                    continue;
                }
				
				$tag = substr(strtoupper(sanitize_title($index)), 0, 10);
				$merge_fields[ $tag ] = (string)$pc_users->data_to_human($index, $val);
			}
			
			// interests (categories)
			$interests = array();
			foreach($cats as $wp_id => $mc_id) {
				$interests[ $mc_id ] = (in_array($wp_id, $ud['categories'])) ? true : false;
			}

			// wrap up
			$data = array(
				'email_address'		=> $ud['email'],
				'status'			=> 'subscribed',
				'merge_fields'		=> $merge_fields,
				'interests'			=> $interests,
				'timestamp_signup'	=> $ud['insert_date'],	
                
                'marketing_permissions' => array(array(
                    'marketing_permission_id' => md5($ud['email']),
                    'enabled' => true,
                )),
			);
            
			$structure['operations'][] = array(
				'path' => 'lists/'. $this->list .'/members/'. md5($ud['email']),
				'method' => 'PUT',
				'body' => json_encode($data)
			);
		}
		
		// only one user - use direct call
		if(count($to_sync) == 1) {
			$url = $this->mc_baseurl. 'lists/'. $this->list .'/members/'. md5($ud['email']);
			$response = $this->call($url, 'PUT', $data);	
		}
	
		else {
			$this->synced_num = count($to_sync);
			
			$url = $this->mc_baseurl. 'batches?skip_merge_validation=true&skip_duplicate_check=true';
			$response = $this->call($url, 'POST', $structure);
		}
		
		if($this->debug_mode) {
			$this->debug_call($response);
		}
		return ($response) ? true : false;
	}
	


	/* 
	 * remove members
	 *
	 * @param (array) $subj - array of users ID to remove
	 * @return (bool)
	 */
	public function remove_members($subj = array()) {
		global $pc_users;
		
		// retrieve users e-mail 
		$args = array(
			'to_get' 	=> array('email'),
			'limit'		=> count((array)$subj),
			'search' 	=> array(
				array('key'=>'email', 'operator'=>'!=', 'val'=>''),
				array('key'=>'id', 'operator'=>'IN', 'val'=>(array)$subj)
			)
		);
		
		$user_query = $pc_users->get_users($args);
		if(!is_array($user_query) || !count($user_query)) {
            return true;
        }
		
	
		// create batch structure 
		$structure = array('operations' => array());
		foreach($user_query as $ud) {
			
			$structure['operations'][] = array(
				'path' => 'lists/'. $this->list .'/members/'. md5($ud['email']),
				'method' => 'DELETE'
			);
		}
	
	
		// only one user - use direct call
		if(count((array)$subj) == 1) {
			$url = $this->mc_baseurl. 'lists/'. $this->list .'/members/'. md5($ud['email']);
			$response = $this->call($url, 'DELETE');	
		}
	
		else {
			$url = $this->mc_baseurl. 'batches';
			$response = $this->call($url, 'POST', $structure);
		}


		if($this->debug_mode) {
			$this->debug_call($response);
		}
		return ($response) ? true : false;
	}
}
